import Input from '@/components/ui/input';
import {
  Control,
  FieldErrors,
  useForm,
  useFormState,
  useWatch,
} from 'react-hook-form';
import Button from '@/components/ui/button';
import TextArea from '@/components/ui/text-area';
import Label from '@/components/ui/label';
import Card from '@/components/common/card';
import Description from '@/components/ui/description';
import * as categoriesIcon from '@/components/icons/category';
import { getIcon } from '@/utils/get-icon';
import { useRouter } from 'next/router';
import ValidationError from '@/components/ui/form-validation-error';
import { useEffect } from 'react';
import { AttachmentInput, Category, Menu } from '@/types';
import { menuIcons } from './menu-icons';
import { useTranslation } from 'next-i18next';
import FileInput from '@/components/ui/file-input';
import SelectInput from '@/components/ui/select-input';
import { yupResolver } from '@hookform/resolvers/yup';
import { menuValidationSchema } from './menu-validation-schema';

import { useTypesQuery } from '@/data/type';
import {
  useCreateMenuMutation,
  useMenusQuery,
  useUpdateMenuMutation,
} from '@/data/menu';

export const updatedIcons = menuIcons.map((item: any) => {
  item.label = (
    <div className="flex items-center space-s-5">
      <span className="flex h-5 w-5 items-center justify-center">
        {getIcon({
          iconList: categoriesIcon,
          iconName: item.value,
          className: 'max-h-full max-w-full',
        })}
      </span>
      <span>{item.label}</span>
    </div>
  );
  return item;
});

type FormValues = {
  title: string;
  link: string;
  parent_id: Menu;
  // type: any;
};

const defaultValues = {
  title: '',
  link: '',
  parent_id: '',
};

type IProps = {
  initialValues?: Menu | undefined;
};
export default function MenuForm({ initialValues }: IProps) {
  const router = useRouter();
  const { t } = useTranslation();
  const isNewTranslation = router?.query?.action === 'translate';
  const {
    register,
    handleSubmit,
    control,
    setValue,

    formState: { errors },
  } = useForm<FormValues>({
    // shouldUnregister: true,
    //@ts-ignore
    defaultValues: initialValues
      ? {
          ...initialValues,

          ...(isNewTranslation && {
            type: null,
          }),
        }
      : defaultValues,
    resolver: yupResolver(menuValidationSchema),
  });

  const { mutate: createMenu, isLoading: creating } = useCreateMenuMutation();
  const { mutate: updateMenu, isLoading: updating } = useUpdateMenuMutation();
  const { menus, paginatorInfo, loading, error } = useMenusQuery({
    limit: 200,
  });

  const onSubmit = async (values: FormValues) => {
    const input = {
      title: values.title,
      link: values.link,

      parent_id: values.parent_id?.id ?? null,
      // type_id: values.type?.id,
    };

    if (!initialValues) {
      createMenu({
        ...input,
        ...(initialValues?.slug && { slug: initialValues.slug }),
      });
    } else {
      updateMenu({
        ...input,
        id: initialValues.id!,
      });
    }
  };

  return (
    <form onSubmit={handleSubmit(onSubmit)}>
      <div className="my-5 flex flex-wrap sm:my-8">
        <Description
          title={'Menu'}
          className="w-full px-0 pb-5 sm:w-4/12 sm:py-8 sm:pe-4 md:w-1/3 md:pe-5 "
        />

        <Card className="w-full sm:w-8/12 md:w-2/3">
          <Input
            label="Title"
            {...register('title')}
            error={t(errors.title?.message!)}
            variant="outline"
            className="mb-5"
          />

          <div className="mb-5">
            <Label>Parent Menu</Label>
            <SelectInput
              name="parent_id"
              control={control}
              options={menus}
              isClearable={true}
              isMulti={false}
              getOptionLabel={(option: any) => option.title}
              getOptionValue={(option: any) => option.id}
            />
          </div>

          <Input
            label="Link"
            {...register('link')}
            error={t(errors.link?.message!)}
            variant="outline"
            className="mb-5"
          />
        </Card>
      </div>
      <div className="mb-4 text-end">
        {initialValues && (
          <Button
            variant="outline"
            onClick={router.back}
            className="me-4"
            type="button"
          >
            {t('form:button-label-back')}
          </Button>
        )}

        <Button loading={creating || updating}>
          {initialValues ? 'Update Menu' : 'Create Menu'}
        </Button>
      </div>
    </form>
  );
}
