import Router, { useRouter } from 'next/router';
import { toast } from 'react-toastify';
import { useTranslation } from 'next-i18next';
import { useQuery, useMutation, useQueryClient } from 'react-query';
import { API_ENDPOINTS } from '@/data/client/api-endpoints';
import { bannerClient } from './client/banner';
import {
  BannerQueryOptions,
  GetParams,
  BannerPaginator,
  Banner,
} from '@/types';
import { mapPaginatorData } from '@/utils/data-mappers';
import { Routes } from '@/config/routes';
import { Config } from '@/config';

export const useCreateBannerMutation = () => {
  const queryClient = useQueryClient();
  const router = useRouter();
  const { t } = useTranslation();
  return useMutation(bannerClient.create, {
    onSuccess: async () => {
      const generateRedirectUrl = router.query.shop
        ? `/${router.query.shop}${Routes.banner.list}`
        : Routes.banner.list;
      await Router.push(generateRedirectUrl, undefined, {
        locale: Config.defaultLanguage,
      });
      toast.success(t('common:successfully-created'));
    },
    // Always refetch after error or success:
    onSettled: () => {
      queryClient.invalidateQueries(API_ENDPOINTS.BANNERS);
    },
  });
};

export const useUpdateBannerMutation = () => {
  const { t } = useTranslation();
  const queryClient = useQueryClient();
  const router = useRouter();
  return useMutation(bannerClient.update, {
    onSuccess: async (data) => {
      const generateRedirectUrl = Routes.banner.list;
      await router.push(generateRedirectUrl);
      toast.success(t('common:successfully-updated'));
    },
    // Always refetch after error or success:
    onSettled: () => {
      queryClient.invalidateQueries(API_ENDPOINTS.BANNERS);
    },
  });
};

export const useDeleteBannerMutation = () => {
  const queryClient = useQueryClient();
  const { t } = useTranslation();
  return useMutation(bannerClient.delete, {
    onSuccess: () => {
      toast.success(t('common:successfully-deleted'));
    },
    // Always refetch after error or success:
    onSettled: () => {
      queryClient.invalidateQueries(API_ENDPOINTS.BANNERS);
    },
  });
};

export const useBannerQuery = ({ slug, language }: GetParams) => {
  const { data, error, isLoading } = useQuery<Banner, Error>(
    [API_ENDPOINTS.BANNERS, { slug, language }],
    () => bannerClient.get({ slug, language })
  );

  return {
    banner: data,
    error,
    isLoading,
  };
};

export const useBannersQuery = (
  params: Partial<BannerQueryOptions>,
  options: any = {}
) => {
  const { data, error, isLoading } = useQuery<BannerPaginator, Error>(
    [API_ENDPOINTS.BANNERS, params],
    ({ queryKey, pageParam }) =>
      bannerClient.paginated(Object.assign({}, queryKey[1], pageParam)),
    {
      keepPreviousData: true,
      ...options,
    }
  );

  return {
    banners: data?.data?.data ?? [],
    paginatorInfo: mapPaginatorData(data),
    error,
    loading: isLoading,
  };
};
