import Router, { useRouter } from 'next/router';
import { toast } from 'react-toastify';
import { useTranslation } from 'next-i18next';
import { useQuery, useMutation, useQueryClient } from 'react-query';
import { API_ENDPOINTS } from '@/data/client/api-endpoints';
import { carouselClient } from './client/carousel';
import {
  CarouselQueryOptions,
  GetParams,
  CarouselPaginator,
  Carousel,
} from '@/types';
import { mapPaginatorData } from '@/utils/data-mappers';
import { Routes } from '@/config/routes';
import { Config } from '@/config';

export const useCreateCarouselMutation = () => {
  const queryClient = useQueryClient();
  const router = useRouter();
  const { t } = useTranslation();
  return useMutation(carouselClient.create, {
    onSuccess: async () => {
      const generateRedirectUrl = router.query.shop
        ? `/${router.query.shop}${Routes.carousel.list}`
        : Routes.carousel.list;
      await Router.push(generateRedirectUrl, undefined, {
        locale: Config.defaultLanguage,
      });
      toast.success(t('common:successfully-created'));
    },
    // Always refetch after error or success:
    onSettled: () => {
      queryClient.invalidateQueries(API_ENDPOINTS.CAROUSELS);
    },
  });
};

export const useUpdateCarouselMutation = () => {
  const { t } = useTranslation();
  const queryClient = useQueryClient();
  const router = useRouter();
  return useMutation(carouselClient.update, {
    onSuccess: async (data) => {
      router.push('/carousels');
      // const generateRedirectUrl = router.query.shop
      //   ? `/${router.query.shop}${Routes.carousel.list}`
      //   : Routes.carousel.list;
      // await router.push(`${generateRedirectUrl}/${data?.id}/edit`, undefined, {
      //   locale: Config.defaultLanguage,
      // });
      toast.success(t('common:successfully-updated'));
    },
    // Always refetch after error or success:
    onSettled: () => {
      queryClient.invalidateQueries(API_ENDPOINTS.CAROUSELS);
    },
  });
};

export const useDeleteCarouselMutation = () => {
  const queryClient = useQueryClient();
  const { t } = useTranslation();
  return useMutation(carouselClient.delete, {
    onSuccess: () => {
      toast.success(t('common:successfully-deleted'));
    },
    // Always refetch after error or success:
    onSettled: () => {
      queryClient.invalidateQueries(API_ENDPOINTS.CAROUSELS);
    },
  });
};

export const useCarouselQuery = ({ slug, language }: GetParams) => {
  const { data, error, isLoading } = useQuery<Carousel, Error>(
    [API_ENDPOINTS.CAROUSELS, { slug, language }],
    () => carouselClient.get({ slug, language })
  );

  return {
    carousel: data,
    error,
    isLoading,
  };
};

export const useCarouselsQuery = (
  params: Partial<CarouselQueryOptions>,
  options: any = {}
) => {
  const { data, error, isLoading } = useQuery<CarouselPaginator, Error>(
    [API_ENDPOINTS.CAROUSELS, params],
    ({ queryKey, pageParam }) =>
      carouselClient.paginated(Object.assign({}, queryKey[1], pageParam)),
    {
      keepPreviousData: true,
      ...options,
    }
  );

  return {
    carousels: data?.data?.data ?? [],
    paginatorInfo: mapPaginatorData(data),
    error,
    loading: isLoading,
  };
};
