import Router, { useRouter } from 'next/router';
import { useMutation, useQuery, useQueryClient } from 'react-query';
import { toast } from 'react-toastify';
import { useTranslation } from 'next-i18next';
import { Routes } from '@/config/routes';
import { API_ENDPOINTS } from './client/api-endpoints';
import { Menu, MenuPaginator, MenuQueryOptions, GetParams } from '@/types';
import { mapPaginatorData } from '@/utils/data-mappers';
import { menuClient } from './client/menu';
import { Config } from '@/config';

export const useCreateMenuMutation = () => {
  const queryClient = useQueryClient();
  const { t } = useTranslation();

  return useMutation(menuClient.create, {
    onSuccess: () => {
      Router.push(Routes.menu.list, undefined, {
        locale: Config.defaultLanguage,
      });
      toast.success(t('common:successfully-created'));
    },
    // Always refetch after error or success:
    onSettled: () => {
      queryClient.invalidateQueries(API_ENDPOINTS.MENUS);
    },
  });
};

export const useDeleteMenuMutation = () => {
  const queryClient = useQueryClient();
  const { t } = useTranslation();

  return useMutation(menuClient.delete, {
    onSuccess: () => {
      toast.success(t('common:successfully-deleted'));
    },
    // Always refetch after error or success:
    onSettled: () => {
      queryClient.invalidateQueries(API_ENDPOINTS.MENUS);
    },
  });
};

export const useUpdateMenuMutation = () => {
  const { t } = useTranslation();
  const queryClient = useQueryClient();
  return useMutation(menuClient.update, {
    onSuccess: () => {
      toast.success(t('common:successfully-updated'));
      Router.push(Routes.menu.list);
    },
    // Always refetch after error or success:
    onSettled: () => {
      queryClient.invalidateQueries(API_ENDPOINTS.MENUS);
    },
  });
};

export const useMenuQuery = ({ slug, language }: GetParams) => {
  const { data, error, isLoading } = useQuery<Menu, Error>(
    [API_ENDPOINTS.MENUS, { slug, language }],
    () => menuClient.get({ slug, language })
  );

  return {
    menu: data?.data,
    error,
    isLoading,
  };
};

export const useMenusQuery = (options: Partial<MenuQueryOptions>) => {
  const { data, error, isLoading } = useQuery<MenuPaginator, Error>(
    [API_ENDPOINTS.MENUS, options],
    ({ queryKey, pageParam }) =>
      menuClient.paginated(Object.assign({}, queryKey[1], pageParam)),
    {
      keepPreviousData: true,
    }
  );

  return {
    menus: data?.data?.data ?? [],
    paginatorInfo: mapPaginatorData(data),
    error,
    loading: isLoading,
  };
};
