import Router, { useRouter } from 'next/router';
import { toast } from 'react-toastify';
import { useTranslation } from 'next-i18next';
import { useQuery, useMutation, useQueryClient } from 'react-query';
import { API_ENDPOINTS } from '@/data/client/api-endpoints';
import { SectionClient } from './client/section';
import { GetParams, Section, SectionPaginator } from '@/types';
import { mapPaginatorData } from '@/utils/data-mappers';
import { Routes } from '@/config/routes';
import { Config } from '@/config';

export const useUpdateSectionMutation = () => {
  const { t } = useTranslation();
  const queryClient = useQueryClient();
  const router = useRouter();
  return useMutation(SectionClient.update, {
    onSuccess: async (data) => {
      await router.push(Routes.section.list, undefined, {
        locale: Config.defaultLanguage,
      });
      toast.success(t('common:successfully-updated'));
    },
    // Always refetch after error or success:
    onSettled: () => {
      queryClient.invalidateQueries(API_ENDPOINTS.SECTION);
    },
  });
};

export const useSectionQuery = (options: any = {}) => {
  const { data, error, isLoading } = useQuery<SectionPaginator, Error>(
    [API_ENDPOINTS.SECTION],
    ({ queryKey, pageParam }) =>
      SectionClient.paginated(Object.assign({}, queryKey[1], pageParam)),
    {
      keepPreviousData: true,
      ...options,
    }
  );

  return {
    sections: data?.data ?? [],
    paginatorInfo: mapPaginatorData(data),
    error,
    loading: isLoading,
  };
};

export const useSingleSectionQuery = ({ slug, language }: GetParams) => {
  const { data, error, isLoading } = useQuery<Section, Error>(
    [API_ENDPOINTS.SECTION, { slug, language }],
    () => SectionClient.get({ slug, language })
  );

  return {
    section: data,
    error,
    isLoading,
  };
};
