import Router, { useRouter } from 'next/router';
import { toast } from 'react-toastify';
import { useTranslation } from 'next-i18next';
import { useQuery, useMutation, useQueryClient } from 'react-query';
import { API_ENDPOINTS } from '@/data/client/api-endpoints';
import { shippingAddressClient } from './client/shipping-address';
import {
  ShippingAddressQueryOptions,
  GetParams,
  ShippingAddressPaginator,
  ShippingAddress,
} from '@/types';
import { mapPaginatorData } from '@/utils/data-mappers';
import { Routes } from '@/config/routes';
import { Config } from '@/config';

export const useCreateShippingAddressMutation = () => {
  const queryClient = useQueryClient();
  const router = useRouter();
  const { t } = useTranslation();
  return useMutation(shippingAddressClient.create, {
    onSuccess: async () => {
      // const generateRedirectUrl = router.query.shop
      //   ? `/${router.query.shop}${Routes.shippingAddress.list}`
      //   : Routes.shippingAddress.list;
      // await Router.push(generateRedirectUrl, undefined, {
      //   locale: Config.defaultLanguage,
      // });

      router.push(Routes['shipping-address'].list);
      toast.success(t('common:successfully-created'));
    },
    // Always refetch after error or success:
    onSettled: () => {
      queryClient.invalidateQueries(API_ENDPOINTS.SHIPPING_ADDRESS);
    },
  });
};

export const useUpdateShippingAddressMutation = () => {
  const { t } = useTranslation();
  const queryClient = useQueryClient();
  const router = useRouter();
  return useMutation(shippingAddressClient.update, {
    onSuccess: async (data) => {
      router.push('/shipping-address');
      // const generateRedirectUrl = router.query.shop
      //   ? `/${router.query.shop}${Routes.shippingAddress.list}`
      //   : Routes.shippingAddress.list;
      // await router.push(`${generateRedirectUrl}/${data?.id}/edit`, undefined, {
      //   locale: Config.defaultLanguage,
      // });
      toast.success(t('common:successfully-updated'));
    },
    // Always refetch after error or success:
    onSettled: () => {
      queryClient.invalidateQueries(API_ENDPOINTS.CAROUSELS);
    },
  });
};

export const useDeleteShippingAddressMutation = () => {
  const queryClient = useQueryClient();
  const { t } = useTranslation();
  return useMutation(shippingAddressClient.delete, {
    onSuccess: () => {
      toast.success(t('common:successfully-deleted'));
    },
    // Always refetch after error or success:
    onSettled: () => {
      queryClient.invalidateQueries(API_ENDPOINTS.SHIPPING_ADDRESS);
    },
  });
};

export const useSingleShippingAddressQuery = ({
  slug,
  language,
}: GetParams) => {
  const { data, error, isLoading } = useQuery<ShippingAddress, Error>(
    [API_ENDPOINTS.SHIPPING_ADDRESS, { slug, language }],
    () => shippingAddressClient.get({ slug, language })
  );

  return {
    shippingAddress: data,
    error,
    isLoading,
  };
};

export const useShippingAddressQuery = (
  params: Partial<ShippingAddressQueryOptions>,
  options: any = {}
) => {
  const { data, error, isLoading } = useQuery<ShippingAddressPaginator, Error>(
    [API_ENDPOINTS.SHIPPING_ADDRESS, params],
    ({ queryKey, pageParam }) =>
      shippingAddressClient.paginated(
        Object.assign({}, queryKey[1], pageParam)
      ),
    {
      keepPreviousData: true,
      ...options,
    }
  );

  return {
    addresses: data?.data?.data ?? [],
    paginatorInfo: mapPaginatorData(data),
    error,
    loading: isLoading,
  };
};
