import Router, { useRouter } from 'next/router';
import { toast } from 'react-toastify';
import { useTranslation } from 'next-i18next';
import { useQuery, useMutation, useQueryClient } from 'react-query';
import { API_ENDPOINTS } from '@/data/client/api-endpoints';
import { sixAdBannerClient } from './client/six-ad-banner';
import { GetParams, SixAdBanner, SixAdBannerPaginator } from '@/types';
import { mapPaginatorData } from '@/utils/data-mappers';
import { Routes } from '@/config/routes';
import { Config } from '@/config';

export const useUpdateSixBannerMutation = () => {
  const { t } = useTranslation();
  const queryClient = useQueryClient();
  const router = useRouter();
  return useMutation(sixAdBannerClient.update, {
    onSuccess: async (data) => {
      await router.push(Routes.sixAdBanner.list, {
        locale: Config.defaultLanguage,
      });
      toast.success(t('common:successfully-updated'));
    },
    // Always refetch after error or success:
    onSettled: () => {
      queryClient.invalidateQueries(API_ENDPOINTS.BANNERS);
    },
  });
};

export const useSixAdBannerQuery = ({ slug, language }: GetParams) => {
  const { data, error, isLoading } = useQuery<SixAdBanner, Error>(
    [API_ENDPOINTS.CAROUSELS, { slug, language }],
    () => sixAdBannerClient.get({ slug, language })
  );

  return {
    banner: data,
    error,
    isLoading,
  };
};

export const useSixAdBannersQuery = (options: any = {}) => {
  const { data, error, isLoading } = useQuery<SixAdBannerPaginator, Error>(
    [API_ENDPOINTS.SIXADBANNERS],
    ({ queryKey, pageParam }) =>
      sixAdBannerClient.paginated(Object.assign({}, queryKey[1], pageParam)),
    {
      keepPreviousData: true,
      ...options,
    }
  );

  return {
    banners: data?.data ?? [],
    paginatorInfo: mapPaginatorData(data),
    error,
    loading: isLoading,
  };
};
